/*
 * File:       MIDI_keyboard(.ino)
 *
 * Project:    MIDI Controller Keyboard -- Ref: www.mjbauer.biz
 *
 * Platform:   Sparkfun 'Pro Micro' dev board (ATmega32U4, 5V/16MHz)
 *             NB: Clones may appear on Arduino IDE as 'Leonardo'
 *
 * Author:     M.J.Bauer, 2026 -- 
 *
 * Licence:    Open Source (Unlicensed) -- free to copy, distribute, modify
 *
 * Revision History:
 * `````````````````
 *   v0.95 :   Pedals (external controls) not yet supported... (TBD v1.0)
 *             'High-Note Priority' (option) and CV/Gate operation not fully tested.
 *             Key contacts wired in 8x8 matrix (build option) not tested.
 *
 */
#include <SPI.h>

#define FIRMWARE_VERSION   "0.95"

#ifndef FALSE
#define FALSE  0
#define TRUE  (!FALSE)
#endif

//-------------------  F I R M W A R E   B U I L D   O P T I O N S  ---------------------
//
#define USE_HEARTBEAT_LED       TRUE   // Set TRUE to enable TX LED flash (@ 2Hz)
#define USE_SERVICE_PORT        TRUE   // Set TRUE to include Service Port code
#define CV_HIGH_NOTE_PRIORITY   FALSE  // Set TRUE for Highest-Note Priority
#define KEY_CONTACTS_IN_MATRIX  FALSE  // Set TRUE if key contacts wired in 2D matrix
#define NUMBER_OF_KEYS          44     // Number of keys on keyboard
#define LOWEST_KEY_NOTE_NUMBER  41     // Lowest note on keyboard (MIDI note number)
#define CV1_OFFSET_SEMITONES     5     // Added to key number to derive CV1 volts
#define MODULATION_CC_INTERVAL  50     // Minimum time between MIDI CC01 messages (ms)
#define DEFAULT_VELOCITY_LEVEL  64     // In case velocity is not variable 
//
// see also: MidiChannelsList[] below to define your set of MIDI channels
//---------------------------------------------------------------------------------------

#define OFF    0
#define ON     1
#define NOT_BUSY      0xFF  // ActiveNotesBuffer[] value
#define INVALID       0xFF  // keyLastGated value (if none)
#define NOTE_OFF_CMD  0x80  // MIDI message status byte
#define NOTE_ON_CMD   0x90  // MIDI message status byte
#define PRIORITY      'P'   // keying mode for CV/Gate
#define LEGATO        'L'   // keying mode for CV/Gate
#define MODULATION    'M'   // knob function 1
#define VELOCITY      'V'   // knob function 2
#define OCTAVE_UP     'U'   // octave shift = (+1)
#define OCTAVE_DOWN   'D'   // octave shift = (-1)

// MCU I/O pin assignments
#define SUSTAIN_SW     2    // PD1 = sustain switch (active low)
#define GATE_OUT       3    // PD0 = GATE output pin (active high)
#define PEDAL_DET      5    // PC6 = pedal detect pin
#define CV_MODE_SW     6    // PD7 = CV/Gate mode (Priority or Legato)
#define MODN_VELO_SW   7    // PE6 = modulation (knob = high, pedal = low)
#define SPI_SH_LD      8    // PB4 = SPI SH/LD (shift = high, load = low)
#define CV1_OUT        9    // OC1A = CV1, PWM output
#define CV2_OUT       10    // OC1B = CV2, PWM output

#define PIN_MODE_PB0_OUTPUT()   (DDRB = DDRB | (1<<0))  // PB0 = RXLED
#define TURN_OFF_RXLED()        (PORTB = PORTB | (1<<0))
#define TURN_ON_RXLED()         (PORTB = PORTB & ~(1<<0))  // ON = LOW
#define PIN_MODE_PD5_OUTPUT()   (DDRD = DDRD | (1<<5))  // PD5 = TXLED
#define TURN_OFF_TXLED()        (PORTD = PORTD | (1<<5))
#define TURN_ON_TXLED()         (PORTD = PORTD & ~(1<<5))  // ON = LOW

#define PIN_MODE_PD2_INPUT()    { PORTD = PORTD | (1<<2);  DDRD = DDRD & ~(1<<2); }
#define TEST_BUTTON_IS_PRESSED    ((PIND & (1<<2)) == 0)  // 'TEST' button is PD2/RX
#define Set_CV1_output(duty)    TC1_OC1A_UpdateDuty((uint16_t)(duty))  // 100% FS = 480
#define Set_CV2_output(duty)    TC1_OC1B_UpdateDuty((uint16_t)(duty))  // 100% FS = 480
#define GateOn()    digitalWrite(GATE_OUT, HIGH)
#define GateOff()   digitalWrite(GATE_OUT, LOW)

// Choose any 5 channels in the range 1 to 16 corresponding to the 5 switch positions:
uint8_t  MidiChannelsList[] = { 1, 3, 5, 7, 9 };

uint8_t  ActiveNotesBuffer[16]; 
uint8_t  keyDriveBit;        // bit position in KeyDriveLines
uint8_t  keyScanOctet;       // key scan sequence variable
uint8_t  MidiChannel;        // MIDI channel setting: 1..16
uint8_t  CV_Gate_mode;       // PRIORITY or LEGATO
uint8_t  keyLastGated;       // KEY number, NOT MIDI note
uint8_t  knobFunction;       // MODULATION or VELOCITY
uint8_t  pedalFunction;      // VELOCITY or MODULATION (inverse of knob fn)
uint8_t  octaveShift;        // NONE (0) or OCTAVE_UP or OCTAVE_DOWN
uint8_t  SPI_databyte;       // debug usage only
long     knobReadingFilter;  // average of knob ADC readings [24:8 bits]

//---------------------------------------------------------------------------------------

void  setup() 
{
  pinMode(SUSTAIN_SW, INPUT_PULLUP);
  pinMode(PEDAL_DET, INPUT_PULLUP);
  pinMode(MODN_VELO_SW, INPUT_PULLUP);
  pinMode(CV_MODE_SW, INPUT_PULLUP);
  pinMode(SPI_SH_LD, OUTPUT);
  pinMode(GATE_OUT, OUTPUT);
  pinMode(CV1_OUT, OUTPUT);
  pinMode(CV2_OUT, OUTPUT);
  PIN_MODE_PB0_OUTPUT();
  PIN_MODE_PD5_OUTPUT();
  
  TC1_Setup_PWM();  // Initialize Timer-Counter #1 for CV outputs
  Serial1.begin(31250);  // Initialize UART (TX) for MIDI OUT
  PIN_MODE_PD2_INPUT();  // Must follow Serial1.begin()! (PD2 = RX)
  GateOff();
  UpdateOperationalModes();  // Read front-panel switches
  ClearActiveNotesBuffer();
  SPI.begin();  // Init. SPI for key contact scanning

#if KEY_CONTACTS_IN_MATRIX
  // Activate initial keyboard drive line (bit0 set low)
  SPI.beginTransaction(SPISettings(4000000, LSBFIRST, SPI_MODE0));
  digitalWrite(SPI_SH_LD, LOW);  // assert SS/
  SPI.transfer(0xFE);  // set kbd drive register b0 low
  digitalWrite(SPI_SH_LD, HIGH);  // negate SS/
  SPI.endTransaction();
  keyDriveBit = 1;  // for next reading
#else  // using SPI input port expander
  digitalWrite(SPI_SH_LD, LOW);  // parallel load shift-registers
#endif  
  
#if USE_SERVICE_PORT
  Serial.begin(57600);  // Initialize USB-serial for Service Port
#endif
}


void  loop() 
{
  static uint32_t startPeriod_5ms, startPeriod_50ms, startPeriod_500ms;
  static uint32_t startModnCCinterval; 
  uint8_t  data7bits;  // 7-bit data value
  
  KeyboardScanRoutine();

#if KEY_CONTACTS_IN_MATRIX
  if (keyDriveBit == 0) KnobPositionUpdate();  // Sync with KeyboardScanRoutine
#else  // using SPI input port expander
  if (keyScanOctet == 0) KnobPositionUpdate();  // Sync with KeyboardScanRoutine
#endif

  if ((millis() - startPeriod_5ms) >= 5)  // 5ms period ended  
  {
    startPeriod_5ms = millis(); 
    // todo:  Monitor 'TEST' button (if needed)  *************
    //
  }

  if ((millis() - startPeriod_50ms) >= 50)  // 50ms period ended
  {
    startPeriod_50ms = millis(); 
    TURN_OFF_TXLED();  // Heartbeat
    // todo:  Monitor Hold/Sustain foot-switch   *************
    //
  }

  if ((millis() - startPeriod_500ms) >= 500)  // 500ms period ended
  {
    startPeriod_500ms = millis();
    if (USE_HEARTBEAT_LED) TURN_ON_TXLED();
    UpdateOperationalModes();  // Call interval is 0.5 second
  }

  if ((millis() - startModnCCinterval) >= MODULATION_CC_INTERVAL) 
  {
    startModnCCinterval = millis();
    if (knobFunction == MODULATION && KnobMoved())  
    {
      data7bits = (uint8_t) (KnobPosition() >> 3);  // MSB (high-order 7 bits)
      MIDI_SendControlChange(MidiChannel, 0x01, data7bits);
      data7bits = (uint8_t) (KnobPosition() << 4);  // LSB (low-order 7 bits)
      MIDI_SendControlChange(MidiChannel, 0x21, data7bits);
      Set_CV2_output(KnobPosition() / 2);  // full-scale = 480 for 5.0V output
    }
  }

#if USE_SERVICE_PORT
  ServicePortRoutine();
#endif
}


/*
* This function is called at 0.5 second intervals. It monitors the front-panel switches.
* If a change is detected, the operational mode, or MIDI channel, is set accordingly.
*/
void  UpdateOperationalModes()
{
  uint16_t  adcReading;
  uint8_t  swpos;

  // Set MIDI OUT channel according to the channel-select switch pos'n
  adcReading = analogRead(A3);
  if (adcReading < 85) swpos = 0;
  else if (adcReading < 230) swpos = 1;
  else if (adcReading < 340) swpos = 2;
  else if (adcReading < 700) swpos = 3;
  else  swpos = 4;
  MidiChannel = MidiChannelsList[swpos];
  // Determine Knob function
  knobFunction = digitalRead(MODN_VELO_SW) == 0 ? VELOCITY : MODULATION;
  if (knobFunction == MODULATION) pedalFunction = VELOCITY;
  else  pedalFunction = MODULATION;  // opposite of Knob function
  // Determine CV/Gate mode
  swpos = digitalRead(CV_MODE_SW);  // 0 = ON;  1 = OFF (open)
  if (swpos == 0 && CV_Gate_mode != LEGATO)
  {
    CV_Gate_mode = LEGATO;  // switch pos'n has changed since last read
    MIDI_SendControlChange(MidiChannel, 123, 0);  // All notes off
    ClearActiveNotesBuffer();
  }
  if (swpos != 0 && CV_Gate_mode != PRIORITY)
  {
    CV_Gate_mode = PRIORITY;  // switch pos'n has changed since last read
    MIDI_SendControlChange(MidiChannel, 123, 0);  // All notes off
    ClearActiveNotesBuffer();
  }
  // Determine octave shift, but do not update if any keys are pressed
  adcReading = analogRead(A6);
  if (isNoteBufferEmpty())
  {
    if (adcReading < 250) octaveShift = OCTAVE_DOWN;
    else if (adcReading < 750) octaveShift = OCTAVE_UP;
    else  octaveShift = 0;  // Normal (no shift)
  }
}


/*
* This routine is called frequently from the main loop. It reads the states of the
* key contacts. On each call, a different group of 8 contacts is read and the contact
* states are examined to determine if a new key press event has occurred, or if a
* key release event has occurred. If so, the "event" is actioned by another function.
* Total time to scan all (44) keys is about 1.2ms including execution of other tasks.
*
* Note on usage of 24-bit SPI port expansion boards from AliExpress:
* ``````````````````````````````````````````````````````````````````
* Input pins are labelled on the assumption that the SPI transfer mode is "MSB first", 
* i.e. pin '8' is the MSB (bit 7) of the first byte transferred. Since stage 'H' of the
* 74HC165 is shifted out first, and the serial input (SI) is a "carry in" from the next
* (higher-order) register, it seems more logical to regard stage 'H' as the LS bit and
* to use SPI transfer mode "LSB first". The port expander pins should be re-labelled
* so that the HC165 stage 'H' is the lowest numbered input of each 8-bit register.
* The key scan routine assumes that the key contacts are wired with the lowest numbered
* input pin connected to the key of the lowest note, i.e. the left-most key.
*/
void  KeyboardScanRoutine()
{
  uint8_t  KeyDriveLines, keySenseLines;  // sense register bits (active low)
  uint8_t  inputBit, keySenseBit;  // bit position in keySenseLines
  uint8_t  keyNumber;  // keyNumber begins at 0 (lowest note)
  uint8_t  numberOfOctets;  // Octet = group of 8 successive keys
  uint8_t  bitMask, contactStates;

#if KEY_CONTACTS_IN_MATRIX   // Using 8 x 8 matrix = 64 notes (max.)
  //
  // Delay 10us for key drive lines to settle, then read sense lines;
  _delay_us(10);
  SPI.beginTransaction(SPISettings(4000000, LSBFIRST, SPI_MODE0));
  digitalWrite(SPI_SH_LD, LOW);  // assert SS/
  KeyDriveLines = (1 << keyDriveBit) ^ 0xFF;  // for *next* reading
  keySenseLines = SPI.transfer(KeyDriveLines);  // 8 bits, msb first
  digitalWrite(SPI_SH_LD, HIGH);  // neagte SS/
  SPI.endTransaction();

// Analyze the 8 contact states for a key press or release event
  bitMask = 1;  // LSB (bit0) first
  for (keySenseBit = 0; keySenseBit < 8; keySenseBit++)
  {
    keyNumber = keyDriveBit * 8 + keySenseBit;  // 0..63
    if ((keySenseLines & bitMask) == 0)  // key is pressed
    {
      if (!isNoteActive(keyNumber)) KeyPressAction(keyNumber);
    }
    else if (isNoteActive(keyNumber)) KeyReleaseAction(keyNumber);
    bitMask = bitMask << 1;  // next sense line
  }
  if (++keyDriveBit == 8) keyDriveBit = 0;  // Next drive line
  //
#else  // Using SPI INPUT PORT EXPANDER
  //
  numberOfOctets = (NUMBER_OF_KEYS + 7) / 8;  // constant
  if (keyScanOctet == 0)  // begin read cycle
  {
    SPI.beginTransaction(SPISettings(4000000, LSBFIRST, SPI_MODE0));
    digitalWrite(SPI_SH_LD, HIGH);  // enable shift, disable load
  }

  contactStates = SPI.transfer(0xFF);  // read (next) 8 bits
  // Analyze the 8 contact states for a key press or release event
  bitMask = (1 << 0);  // LSB (bit0) first 
  for (inputBit = 0;  inputBit < 8;  inputBit++)
  {
    keyNumber = keyScanOctet * 8 + inputBit;  // range 0 to (NUMBER_OF_KEYS - 1)
    if ((contactStates & bitMask) == 0)  // key is pressed
    {
      if (!isNoteActive(keyNumber)) KeyPressAction(keyNumber);
    }
    else if (isNoteActive(keyNumber)) KeyReleaseAction(keyNumber);
    bitMask = bitMask << 1;  // next bit
  }

  if (++keyScanOctet >= numberOfOctets)  // next octet
  {
    keyScanOctet = 0;  // reset to first octet
    digitalWrite(SPI_SH_LD, LOW);  // done all keys... enable parallel load
    SPI.endTransaction();
  }
  #endif
}

// Function called whenever a key press event is detected...
//
void  KeyPressAction(uint8_t keyNumber)
{
  static uint8_t  lastSlotUsed;  // last buffer slot allocated a note
  uint8_t  slot = lastSlotUsed;
  uint8_t  noteNumber = LOWEST_KEY_NOTE_NUMBER + keyNumber;
  uint16_t CV1_PWM_duty = ((uint16_t)(keyNumber + CV1_OFFSET_SEMITONES) * 480) / 60;
  uint8_t  noteAdjusted;  // according to octaveShift
  uint8_t  velocity7bits = DEFAULT_VELOCITY_LEVEL;
  uint8_t  count = 0;

  if (knobFunction == VELOCITY) velocity7bits = KnobPosition() >> 3;  // 0..127

  while (count++ < 16)  // check 16 slots max.
  {
    if (++slot >= 16)  slot = 0;  // wrrap
    if (ActiveNotesBuffer[slot] == NOT_BUSY)
    {
      ActiveNotesBuffer[slot] = noteNumber;  // slot is now BUSY
      noteAdjusted = noteNumber;
      if (octaveShift == OCTAVE_UP) noteAdjusted += 12;
      if (octaveShift == OCTAVE_DOWN) noteAdjusted -= 12;
      MIDI_SendNoteOn(MidiChannel, noteAdjusted, velocity7bits);
      lastSlotUsed = slot;
      break;
    }
  }

  if (CV_Gate_mode == PRIORITY)
  {
    if (!CV_HIGH_NOTE_PRIORITY || isNoteHighest(keyNumber))
    {
      GateOn();
      keyLastGated = keyNumber;
      Set_CV1_output(CV1_PWM_duty);
      Set_CV2_output((uint16_t)velocity7bits * 4);
    }
  }
  else  // if (CV_Gate_mode == LEGATO) ...
  {
    if (keyLastGated == INVALID)  // no key(s) pressed prior
    {
      GateOn();  // initiate new note (attack)
      keyLastGated = keyNumber;
      Set_CV2_output((uint16_t)velocity7bits * 4);
    }
    Set_CV1_output(CV1_PWM_duty);  // regardless
  }
}

// Function called whenever a key release event is detected...
//
void  KeyReleaseAction(uint8_t keyNumber)
{
  uint8_t  slot;
  uint8_t  noteNumber = LOWEST_KEY_NOTE_NUMBER + keyNumber;
  uint8_t  noteAdjusted;  // according to octaveShift
  uint16_t CV1_PWM_duty = ((uint16_t)(keyLastGated + CV1_OFFSET_SEMITONES) * 480) / 60;

  for (slot = 0;  slot < 16;  slot++) 
  {
    if (ActiveNotesBuffer[slot] == noteNumber)
    {
      ActiveNotesBuffer[slot] = NOT_BUSY;
      noteAdjusted = noteNumber;
      if (octaveShift == OCTAVE_UP) noteAdjusted += 12;
      if (octaveShift == OCTAVE_DOWN) noteAdjusted -= 12;
      MIDI_SendNoteOff(MidiChannel, noteAdjusted);
    }
  }

  if (CV_Gate_mode == PRIORITY)
  {
    if (keyNumber == keyLastGated)
    {
      GateOff();
      keyLastGated = INVALID;
    }
  }
  else  // if (CV_Gate_mode == LEGATO) ...
  {
    if (isNoteBufferEmpty())  // no key(s) pressed now
    {
      GateOff();
      keyLastGated = INVALID;
    }
    else  Set_CV1_output(CV1_PWM_duty);  // change pitch
  }
}


// The Active Notes Buffer contains MIDI note numbers of up to 16 active (pressed) keys.
// Buffer "slots" (array elements) which have no note allocated are set to 'NOT_BUSY'.
//
void  ClearActiveNotesBuffer()
{
  uint8_t  slot;

  for (slot = 0; slot < 16; slot++) ActiveNotesBuffer[slot] = NOT_BUSY;

  keyLastGated = INVALID;
}


// Function returns TRUE if the specified key (keyNumber) is "active", i.e. the key is
// being held pressed.
//
bool  isNoteActive(uint8_t keyNumber)
{
  uint8_t  slot;
  uint8_t  noteNumber = LOWEST_KEY_NOTE_NUMBER + keyNumber;
  uint8_t  status = FALSE;

  for (slot = 0;  slot < 16;  slot++) 
  {
    if (ActiveNotesBuffer[slot] == noteNumber) { status = TRUE;  break; }
  }
  return  status;
}


// Function returns TRUE if the specified key (keyNumber) is higher or the same
// as all currently active (pressed) keys.
//
bool  isNoteHighest(uint8_t keyNumber)
{
  uint8_t  slot;
  uint8_t  noteNumber = LOWEST_KEY_NOTE_NUMBER + keyNumber;
  uint8_t  status = FALSE;

  for (slot = 0;  slot < 16;  slot++) 
  {
    if (noteNumber >= ActiveNotesBuffer[slot]) { status = TRUE;  break; }
  }
  return  status;
}


// Function returns TRUE if there are no active (pressed) keys.
//
bool  isNoteBufferEmpty()
{
  uint8_t  slot;
  uint8_t  status = TRUE;

  for (slot = 0;  slot < 16;  slot++) 
  {
    if (ActiveNotesBuffer[slot] != NOT_BUSY) { status = FALSE;  break; }
  }
  return  status;
}

//========================================================================================

/*
 * Function to read the front-panel control knob analog input.
 * Successive ADC readings are averaged using a 1st-order IIR filter (K = 1/16).
 */
void  KnobPositionUpdate()
{
  long  adcReading;

  adcReading = analogRead(A0);
  adcReading = adcReading << 8;  // convert to fixed-point (8-bit fraction part)
  knobReadingFilter -= (knobReadingFilter >> 4);  // sub (K x knobReadingFilter)
  knobReadingFilter += (adcReading >> 4);  // add (K x adcReading)
}

/*
 * Function returns TRUE if the Knob has been moved more than 10 units (about 1% FS)
 * since a previous call to the function returned TRUE.
 */
bool  KnobMoved()
{
  static long  lastReading;
  bool result = FALSE;

  if (labs(knobReadingFilter - lastReading) > (10 << 8))
  {
    result = TRUE;
    lastReading = knobReadingFilter;
  }
  return result;
}

/*
 * Function returns the modulation/velocity control knob postion as a 10-bit unsigned
 * number in the range 0..1023.
 */
uint16_t  KnobPosition()
{
  return (uint16_t)(knobReadingFilter >> 8);  // integer part
}


/*
 * Function initializes Timer-Counter TC1 in PWM mode to generate two independent 
 * variable-duty pulse waveforms on pins OC1A and OC1B. The PWM carrier frequency and
 * duty resolution are the same on each of the 2 'output compare' channels (A and B).
 * The timer period is set to 480 counts to give a full-scale output voltage of 5.00V. 
 * One semitone increment is equivalent to 8 counts, giving a range of 60 semitones,
 * i.e. 5 octaves, at one volt per octave precisely.
 * The PWM carrier frequency is 33.3 kHz assuming MCU system clock F_OSC = 16 MHz.
 * WGM = 1110 (waveform gen mode 14);  TOP count register is ICR1
 * The timer/counter is enabled by writing TCCR1B with CS[2:0] != 0
 * PWM duty (OCR1A or OCR1B) must be less than the top count register value (ICR1).
 */
void  TC1_Setup_PWM()
{
  TCCR1B = 0;  // Disable timer/counter during setup (CS = 000)
  TCCR1C = 0;  // FOC[2:0] = 0b000
  OCR1A = 0;   // Initial duty = 0
  OCR1B = 0;   // Initial duty = 0
  ICR1 = 479;  // TOP count = 479 for period = 480 counts
  TCCR1A = 0b10100010;  // COM1A = COM1B = 0b10;  WGM[1:0] = 0b10
  TCCR1B = 0b00011001;  // WGM[3:2] = 11;  CS[2:0] = 001 (no prescaler)
}

void  TC1_OC1A_UpdateDuty(uint16_t duty_clks)
{
  if (duty_clks > 478) duty_clks = 478;  // max. duty < ICR1
  OCR1A = duty_clks;
}

void  TC1_OC1B_UpdateDuty(uint16_t duty_clks)
{
  if (duty_clks > 478) duty_clks = 478;  // max. duty < ICR1
  OCR1B = duty_clks;
}


//========================================================================================
//=================  M I D I    M E S S A G I N G    F U N C T I O N S  ==================

void  MIDI_SendNoteOn(uint8_t chan, uint8_t noteNum, uint8_t velocity)
{
    uint8_t   statusByte = 0x90 | ((chan - 1) & 0xF);

    Serial1.write(statusByte);
    Serial1.write(noteNum & 0x7F);
    Serial1.write(velocity & 0x7F);
}

void  MIDI_SendNoteOff(uint8_t chan, uint8_t noteNum)
{
    uint8_t   statusByte = 0x80 | ((chan - 1) & 0xF);

    Serial1.write(statusByte);
    Serial1.write(noteNum & 0x7F);
    Serial1.write(64);
}

void  MIDI_SendAfterTouch(uint8_t chan, uint8_t level)
{
    uint8_t   statusByte = 0xD0 | ((chan - 1) & 0xF);

    Serial1.write(statusByte);
    Serial1.write(level & 0x7F);
}

void  MIDI_SendPitchBend(uint8_t chan, unsigned value)
{
    uint8_t   statusByte = 0xE0 | ((chan - 1) & 0xF);

    Serial1.write(statusByte);
    Serial1.write(value & 0x7F);           // 7 LS bits
    Serial1.write((value >> 7) & 0x7F);    // 7 MS bits
}

void  MIDI_SendControlChange(uint8_t chan, uint8_t ctrlNum, uint8_t value)
{
    uint8_t   statusByte = 0xB0 | ((chan - 1) & 0xF);

    Serial1.write(statusByte);
    Serial1.write(ctrlNum & 0x7F);
    Serial1.write(value & 0x7F);
}

void  MIDI_SendProgramChange(uint8_t chan, uint8_t progNum)
{
    uint8_t   statusByte = 0xC0 | ((chan - 1) & 0xF);

    Serial1.write(statusByte);
    Serial1.write(progNum & 0x7F);
}


//========================================================================================
// The following code implements the "Service Port" facility, intended for diagnostics.
//========================================================================================
#if USE_SERVICE_PORT

#define CMD_LINE_MAX_LEN  60   // Max length of command line (chars)
#define CLI_ARG_MAX_LEN   12   // Max length of cmd argument (chars)
#define SPACE      32
#define ASCII_CR   13
#define ASCII_BS    8
#define ASCII_CAN  24
#define ASCII_ESC  27
#define NEWLINE()   Serial.print("\r\n")

char  cmdLine[CMD_LINE_MAX_LEN + 2];   // Command Line buffer
char  cmdName[CLI_ARG_MAX_LEN + 1];    // Command name (string)
char  argStr1[CLI_ARG_MAX_LEN + 1];    // Command argument string #1
char  argStr2[CLI_ARG_MAX_LEN + 1];    // Command argument string #2
const char *cmdErrorMsg = "! Command error";


void  HelpCommand()
{
  Serial.println("Commands:");
  Serial.println("  ver,  buff,  clear,  midi,  knob,  mode,  spi");
  Serial.println("  gate (0/1)");
  Serial.println("  CV# <duty>  (0..480)");
}

void  VerCommand()
{
  Serial.print("Firmware: v");
  Serial.println(FIRMWARE_VERSION);
}

void  BuffCommand()
{
  uint8_t  slot, value;

  if (isNoteBufferEmpty()) Serial.println("Buffer Empty");
  else for (slot = 0;  slot < 16;  slot++)
  {
    value = ActiveNotesBuffer[slot];
    Serial.print(slot);  Serial.print("\t");  
    if (value == NOT_BUSY) Serial.print("--");
    else  Serial.print((int)value);
    NEWLINE();
  }

  Serial.print("Note Gated: ");
  if (keyLastGated != INVALID) Serial.println((int)keyLastGated);
  else  Serial.println("?");
}

void  ClearCommand()
{
  ClearActiveNotesBuffer();
}

void  MidiCommand()
{
  Serial.print("Channel: ");
  Serial.println((int)MidiChannel);
}

void  KnobCommand()
{
  KnobPositionUpdate();
  Serial.println((int)KnobPosition());
}

void  ModeCommand()
{
  if (CV_Gate_mode != 'L' && CV_Gate_mode != 'P') CV_Gate_mode = '?';
  Serial.print("CV_Gate_mode: ");  
  Serial.println((char)CV_Gate_mode);
  Serial.print("CV mode sw: ");
  if (digitalRead(6) == LOW) Serial.println("ON"); 
  else Serial.println("OFF");
  Serial.print("Knob Fn sw: ");
  if (digitalRead(7) == LOW) Serial.println("ON"); 
  else Serial.println("OFF");
  Serial.print("Octave sel: ");
  if (octaveShift == OCTAVE_UP) Serial.print("+1");
  else if (octaveShift == OCTAVE_DOWN) Serial.print("-1");
  else  Serial.print(" 0");
  Serial.print(" (ADC count = ");
  Serial.print((int)analogRead(A6));
  Serial.println(")");
}

void  GateCommand(uint8_t argCount)
{
  if (argCount >= 2)
  {
    if (argStr1[0] == '1')  GateOn();
    else  GateOff();
  }
}

void  CV1_Command(uint8_t argCount)
{
  uint16_t  arg1;

  if (argCount >= 2)
  {
    arg1 = atoi(argStr1);
    TC1_OC1A_UpdateDuty(arg1);
  }
}

void  CV2_Command(uint8_t argCount)
{
  uint16_t  arg1;

  if (argCount >= 2)
  {
    arg1 = atoi(argStr1);
    TC1_OC1B_UpdateDuty(arg1);
  }
}

// The purpose of the SPI command is to check the clock mode (using 'scope on SCK & MISO)
// and to verify the order of bits in the received byte (i.e. LSB first or MSB first).
//
void  SPI_Command()
{
  uint8_t  bitmask = (1 << 7);  // MSB first

  digitalWrite(SPI_SH_LD, LOW);  // load data from first register
  delay(2);
  SPI.beginTransaction(SPISettings(4000000, LSBFIRST, SPI_MODE0));
  digitalWrite(SPI_SH_LD, HIGH);  // enable shift
  SPI_databyte = SPI.transfer(0);  // get first byte after SH/LD set High
  digitalWrite(SPI_SH_LD, LOW); 
  SPI.endTransaction();

  Serial.print("Rx byte (b7..b0):  ");
  while (bitmask)  
  { 
    if (SPI_databyte & bitmask) Serial.print('1');  
    else  Serial.print('0');
    Serial.print(' ');
    bitmask = bitmask >> 1;
  }
  NEWLINE();
}

//------------------------------------------------------------------------------

void  ServicePortRoutine()
{
  uint8_t  offset = 0;   // marker of next argument in cmdLine[]
	uint8_t  argCount;     // Number of cmd "arguments" incl. cmdName
	uint8_t  cmdLineLength = 0;

  if (getString(cmdLine, CMD_LINE_MAX_LEN))  // TRUE => got a command string
  {
    cmdLineLength = strlen(cmdLine);
    cmdName[0] = 0;  // clear cmd string
    argStr1[0] = argStr2[0] = 0;  // clear arg's
    if (cmdLineLength != 0)
    {
      offset = ExtractArg(cmdLine, 0, cmdName);
      argCount = 1;  // assume we have arg[0] = cmdName

      if (offset < cmdLineLength)  // get 1st arg, if any
      {
        offset = ExtractArg(cmdLine, offset, argStr1);  
        argCount++;
      }
      if (offset < cmdLineLength)  // get 2nd arg, if any
      {
        offset = ExtractArg(cmdLine, offset, argStr2);
        argCount++;
      }

      if (strMatch(cmdName, "help"))  HelpCommand();
      else if (strMatch(cmdName, "ver"))  VerCommand();
      else if (strMatch(cmdName, "buff"))  BuffCommand();
      else if (strMatch(cmdName, "clear"))  ClearCommand();
      else if (strMatch(cmdName, "midi"))  MidiCommand();
      else if (strMatch(cmdName, "knob"))  KnobCommand();
      else if (strMatch(cmdName, "mode"))  ModeCommand();
      else if (strMatch(cmdName, "spi"))  SPI_Command();
      else if (strMatch(cmdName, "gate"))  GateCommand(argCount);
      else if (strMatch(cmdName, "cv1"))  CV1_Command(argCount);
      else if (strMatch(cmdName, "cv2"))  CV2_Command(argCount);
      else  Serial.println("! Undefined command");
    }
    Serial.print("> ");  // prompt
  }
}

uint8_t  ExtractArg(char *source, uint8_t offset, char *dest)
{
	uint8_t  index = offset;  // index into input array, source[]
	uint8_t  outdex = 0;      // index into output array, dest[]
	uint8_t  count = 0;

	if (source[index] < SPACE)  return index;  // end-of message -- bail
	while (source[index] == SPACE)  { index++; }  // skip space(s)

	while (count < CLI_ARG_MAX_LEN)   // copy chars to dest[]
	{
		if (source[index] <= SPACE) break;  // control code or space -- bail
		dest[outdex++] = source[index++];   // copy 1 char
		dest[outdex] = 0;  // terminate string
		count++;
	}
	if (source[index] < SPACE)  return index;  // end-of message -- bail
	while (source[index] == SPACE)  { index++; }  // skip space(s)
	return  index;
}

bool  strMatch(char *str1, char *str2)
{
  uint8_t  k;
  char   c1, c2;
  bool   result = TRUE;

  for (k = 0;  k < 255;  k++)
  {
    c1 = tolower( str1[k] );
    c2 = tolower( str2[k] );
    if (c1 != c2)  result = FALSE;
    if (c1 == 0 || c2 == 0)  break;  // found NUL -- exit
  }
  return result;
}

bool  getString(char *str, uint8_t maxlen)
{
	static uint8_t  index;  // index into str[] - saved across calls
	static uint8_t  count;  // number of chars in output string - saved
	char  rxb;              // received byte/character
	bool  status = FALSE;   // return value

	if (Serial.available()) 
	{
		rxb = Serial.read();

		if (rxb == ASCII_CR)   // CR code -- got complete command line
		{
			Serial.write("\r\n");  // Echo CR + LF
			str[index] = 0;  // add NUL terminator
			index = 0; 
			count = 0;
			status = TRUE;
		}
		else if (rxb >= SPACE && count < maxlen) // printable char
		{
			Serial.write(rxb);  // echo rxb back to user
			str[index] = rxb;  // append to buffer
			index++;
			count++;
		}
		else if (rxb == ASCII_BS && count != 0)  // Backspace
		{
			Serial.write(ASCII_BS);  // erase offending char
			Serial.write(SPACE);
			Serial.write(ASCII_BS);  // re-position cursor
			index--;  // remove last char in buffer
			count--;
		}
		else if (rxb == ASCII_CAN || rxb == ASCII_ESC)  // Cancel line
		{
			Serial.print(" ^X^ \n");
			Serial.print("> ");        // prompt
			index = 0;
			count = 0;
		}
	}
	return  status;
}

#endif // USE_SERVICE_PORT
